/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qlayout.h>
#include <qpushbutton.h>
#include <qsize.h>
#include <qgroupbox.h>
#include <qstring.h>

#include <kglobal.h>
#include <klocale.h>
#include <kapplication.h>
#include <kfiledialog.h>
#include <kmessagebox.h>

#include "global.h"
#include "regfile.h"
#include "appenddlg.h"


AppendDlg::AppendDlg(QWidget *parent, const char *name): KDialog(parent, name, true)
{
	QBoxLayout *vbox, *hbox, *svbox;
	QGridLayout *grid;
	QLabel *label;
	QGroupBox *gbox;
	QPushButton *bhelp, *bappend, *bclose;
	QSize size;

	ct_first_ed = ct_last_ed = ct_articles = 0;
	ix_first_ed = ix_last_ed = ix_articles = 0;

	vbox = new QVBoxLayout(this, marginHint());

	gbox = new QGroupBox(i18n("Database Status"), this);

	svbox = new QVBoxLayout(gbox, spacingHint());

	svbox->addSpacing(fontMetrics().height());

	grid = new QGridLayout(3, 4, spacingHint());
	svbox->addLayout(grid);

	label = new QLabel(i18n("First Edition"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 1, AlignVCenter | AlignRight);

	label = new QLabel(i18n("Last Edition"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 2, AlignVCenter | AlignRight);

	label = new QLabel(i18n("Articles"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 3, AlignVCenter | AlignRight);

	label = new QLabel(i18n("c't Magazine"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	ct_first = new QLabel(gbox);
	ct_first->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_first->setMinimumSize(ct_first->sizeHint());
	ct_first->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_first, 1, 1);

	ct_last = new QLabel(gbox);
	ct_last->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_last->setMinimumSize(ct_last->sizeHint());
	ct_last->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_last, 1, 2);

	ct_article_no = new QLabel(gbox);
	ct_article_no->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_article_no->setMinimumSize(ct_article_no->sizeHint());
	ct_article_no->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_article_no, 1, 3);

	label = new QLabel(i18n("iX Magazine"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	ix_first = new QLabel(gbox);
	ix_first->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_first->setMinimumSize(ix_first->sizeHint());
	ix_first->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_first, 2, 1);

	ix_last = new QLabel(gbox);
	ix_last->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_last->setMinimumSize(ix_last->sizeHint());
	ix_last->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_last, 2, 2);

	ix_article_no = new QLabel(gbox);
	ix_article_no->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_article_no->setMinimumSize(ix_article_no->sizeHint());
	ix_article_no->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_article_no, 2, 3);

	svbox->activate();

	vbox->addWidget(gbox);

	vbox->addStretch();

	hbox = new QHBoxLayout(spacingHint());
	vbox->addLayout(hbox);

	bhelp = new QPushButton(i18n("&Help"), this);
	bappend = new QPushButton(i18n("&Append"), this);
	bappend->setDefault(true);
	bclose = new QPushButton(i18n("&Close"), this);
	bclose->setFixedSize(bclose->sizeHint());

	size = bhelp->sizeHint().expandedTo(bappend->sizeHint());
	bhelp->setFixedSize(size);
	bappend->setFixedSize(size);

	hbox->addWidget(bhelp);	
	hbox->addStretch();
	hbox->addWidget(bappend);	
	hbox->addWidget(bclose);	
	
	vbox->activate();

	setCaption(i18n("Append Register File"));

	connect(bhelp, SIGNAL(clicked()), this, SLOT(slotHelp()));
	connect(bappend, SIGNAL(clicked()), this, SLOT(slotAppend()));
	connect(bclose, SIGNAL(clicked()), this, SLOT(close()));

	scanEntries();
}


AppendDlg::~AppendDlg()
{
}


void AppendDlg::slotHelp()
{
	kapp->invokeHelp("append-help");
}


void AppendDlg::slotAppend()
{
	QString s;
	RegFile infile, regfile;
	RegEntry entry;
	int i;

	s = KFileDialog::getOpenFileName(QString::null, i18n("*.CTI *.cti|Heise Register Files (*.cti)"), this, i18n("Open Heise Register File"));
	if (s.isEmpty())
		return;

	infile.setName(s);
	if (!infile.open(IO_ReadOnly)) {
		KMessageBox::sorry(this, i18n("Can't open specified register file!"));
		return;
	}

	if (!infile.scanEntries()) {
		infile.close();
		KMessageBox::sorry(this, i18n("Specified file is not a valid register file!"));
		return;
	}

	regfile.setName(expandPath(REGISTER_FILE));
	if (!regfile.open(IO_ReadWrite | IO_Append)) {
		infile.close();
		KMessageBox::sorry(this, i18n("Can't open internal data base file!"));
		return;
	}
	regfile.scanEntries();


	if (regfile.containsEditions(REG_MAGAZINE_CT, infile.editions(REG_MAGAZINE_CT))) {
		regfile.close();
		infile.close();
		KMessageBox::sorry(this, i18n("Register file contains c't magazine editions\nwhich are already in the data base!"));
		return;
	}

	if (regfile.containsEditions(REG_MAGAZINE_IX, infile.editions(REG_MAGAZINE_IX))) {
		regfile.close();
		infile.close();
		KMessageBox::sorry(this, i18n("Register file contains iX magazine editions\nwhich are already in the data base!"));
		return;
	}

	while (!infile.atEnd()) {
		if (!infile.readEntry(&entry)) {
			KMessageBox::sorry(this, i18n("File read error!"));
			break;
		}

		for (i = 0; i < REG_LINE_COUNT; i++) {
			if (regfile.writeBlock(entry.line(i), (uint)strlen(entry.line(i))) == -1) {
				KMessageBox::sorry(this, i18n("File write error!"));
				break;
			}
		}
	}

	regfile.close();
	infile.close();

	scanEntries();
}


void AppendDlg::scanEntries()
{
	RegFile file(expandPath(REGISTER_FILE));

	if (file.open(IO_ReadOnly)) {
		QString s;

		if (!file.scanEntries()) {

		}
		file.close();

		ct_first_ed = file.firstEdition(REG_MAGAZINE_CT);
		ct_last_ed = file.lastEdition(REG_MAGAZINE_CT);
		ct_articles = file.articles(REG_MAGAZINE_CT);
		ix_first_ed = file.firstEdition(REG_MAGAZINE_IX);
		ix_last_ed = file.lastEdition(REG_MAGAZINE_IX);
		ix_articles = file.articles(REG_MAGAZINE_IX);

		if (ct_articles) {
			s.sprintf("%d/%d", ct_first_ed % 100, ct_first_ed / 100);
			ct_first->setText(s);
			s.sprintf("%d/%d", ct_last_ed % 100, ct_last_ed / 100);
			ct_last->setText(s);
			ct_article_no->setText(KGlobal::locale()->formatNumber((double)ct_articles, 0));
			ct_article_no->setMinimumSize(ct_article_no->sizeHint());
		}
		if (ix_articles) {
			s.sprintf("%d/%d", ix_first_ed % 100, ix_first_ed / 100);
			ix_first->setText(s);
			s.sprintf("%d/%d", ix_last_ed % 100, ix_last_ed / 100);
			ix_last->setText(s);
			ix_article_no->setText(KGlobal::locale()->formatNumber((double)ix_articles, 0));
			ix_article_no->setMinimumSize(ix_article_no->sizeHint());
		}
	}
}
