/***************************************************************************
                          PasswordTest.cpp  -  description
                             -------------------
    begin                : Sat Jul 1 2000
    copyright            : (C) 2000 by Franois Dupoux
    email                : fdupoux@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "PasswordTest.h"
#include "resource.h"
#include "kfilecoder.h"
#include "algo_pc1.h"
#include "misc.h"

// ===========================================================================================================================
void getPasswordTestHashTable(BYTE *cHash, const BYTE *cPassword, const BYTE *cMixer)
{
	const DWORD dwBufferLen = 8192; // number of bytes in g_cPasswordTestData
	BYTE cEncodedBuffer[dwBufferLen];
	BYTE cMixedPassword[ENCODING_LEN+1];
	DWORD dwBufferPos;
	int cCurChar;
	DWORD iFor;
		
	// ****** 1. mix cMixer and the password (the password and the mixer number must have the same size)
	for (iFor=0; iFor < ENCODING_LEN; iFor++)
		cMixedPassword[iFor] = cPassword[iFor] ^ cMixer[iFor];
	
	// ****** 2. encode constant buffer with PC1, using the mixed number
		
	// pc1 variables
	UINT pcSi,pcX1a2, pcX1a0[8], pcI, pcInter, pcCfc, pcCfd;
	BYTE pcKey[ENCODING_LEN+1];
	
	// PC1 Init
	memcpy(pcKey, cMixedPassword, ENCODING_LEN);
	pcSi=0;
	pcX1a2=0;
	pcI=0;
		
	// encode data
	for (dwBufferPos=0L; dwBufferPos < dwBufferLen; dwBufferPos++)
	{	
		cCurChar = g_cPasswordTestData[dwBufferPos];
	
		pc1Assemble(pcX1a0, pcKey, &pcInter, &pcI, &pcX1a2, &pcSi);
		
		pcCfc = pcInter >> 8;
		pcCfd = pcInter & 255; // pcCfc^pcCfd = random byte

		// K ZONE !!!!!!!!!!!!!
		// here the mix of c and pcKey[iFor] is before the encryption of cCurChar

		for (iFor=0; iFor < ENCODING_LEN; iFor++)
		{
			// we mix the plaintext byte with the key
			pcKey[iFor] ^= cCurChar;
		}

		cCurChar ^= (pcCfc ^ pcCfd);
		cEncodedBuffer[dwBufferPos] = cCurChar;
	}
	
	// ****** 3. hash the encoded buffer with a constant key and return it (get the "CRC")
	calculatePC1Hash(cHash, cEncodedBuffer, dwBufferLen, 0);
}

// ===========================================================================================================================
// data made with "NumGen" (see source in the KFileCoder directory) using rand()
// there are 8192 bytes here (8192 numbers between 0 and 255)

BYTE g_cPasswordTestData[] = {
103, 198, 105, 115, 81, 255, 74, 236, 41, 205, 186, 171, 242, 251, 227, 70, 124, 194, 84, 248,
27, 232, 231, 141, 118, 90, 46, 99, 51, 159, 201, 154, 102, 50, 13, 183, 49, 88, 163, 90,
37, 93, 5, 23, 88, 233, 94, 212, 171, 178, 205, 198, 155, 180, 84, 17, 14, 130, 116, 65,
33, 61, 220, 135, 112, 233, 62, 161, 65, 225, 252, 103, 62, 1, 126, 151, 234, 220, 107, 150,
143, 56, 92, 42, 236, 176, 59, 251, 50, 175, 60, 84, 236, 24, 219, 92, 2, 26, 254, 67,
251, 250, 170, 58, 251, 41, 209, 230, 5, 60, 124, 148, 117, 216, 190, 97, 137, 249, 92, 187,
168, 153, 15, 149, 177, 235, 241, 179, 5, 239, 247, 0, 233, 161, 58, 229, 202, 11, 203, 208,
72, 71, 100, 189, 31, 35, 30, 168, 28, 123, 100, 197, 20, 115, 90, 197, 94, 75, 121, 99,
59, 112, 100, 36, 17, 158, 9, 220, 170, 212, 172, 242, 27, 16, 175, 59, 51, 205, 227, 80,
72, 71, 21, 92, 187, 111, 34, 25, 186, 155, 125, 245, 11, 225, 26, 28, 127, 35, 248, 41,
248, 164, 27, 19, 181, 202, 78, 232, 152, 50, 56, 224, 121, 77, 61, 52, 188, 95, 78, 119,
250, 203, 108, 5, 172, 134, 33, 43, 170, 26, 85, 162, 190, 112, 181, 115, 59, 4, 92, 211,
54, 148, 179, 175, 226, 240, 228, 158, 79, 50, 21, 73, 253, 130, 78, 169, 8, 112, 212, 178,
138, 41, 84, 72, 154, 10, 188, 213, 14, 24, 168, 68, 172, 91, 243, 142, 76, 215, 45, 155,
9, 66, 229, 6, 196, 51, 175, 205, 163, 132, 127, 45, 173, 212, 118, 71, 222, 50, 28, 236,
74, 196, 48, 246, 32, 35, 133, 108, 251, 178, 7, 4, 244, 236, 11, 185, 32, 186, 134, 195,
62, 5, 241, 236, 217, 103, 51, 183, 153, 80, 163, 227, 20, 211, 217, 52, 247, 94, 160, 242,
16, 168, 246, 5, 148, 1, 190, 180, 188, 68, 120, 250, 73, 105, 230, 35, 208, 26, 218, 105,
106, 126, 76, 126, 81, 37, 179, 72, 132, 83, 58, 148, 251, 49, 153, 144, 50, 87, 68, 238,
155, 188, 233, 229, 37, 207, 8, 245, 233, 226, 94, 83, 96, 170, 210, 178, 208, 133, 250, 84,
216, 53, 232, 212, 102, 130, 100, 152, 217, 168, 135, 117, 101, 112, 90, 138, 63, 98, 128, 41,
68, 222, 124, 165, 137, 78, 87, 89, 211, 81, 173, 172, 134, 149, 128, 236, 23, 228, 133, 241,
140, 12, 102, 241, 124, 192, 124, 187, 34, 252, 228, 102, 218, 97, 11, 99, 175, 98, 188, 131,
180, 105, 47, 58, 255, 175, 39, 22, 147, 172, 7, 31, 184, 109, 17, 52, 45, 141, 239, 79,
137, 212, 182, 99, 53, 193, 199, 228, 36, 131, 103, 216, 237, 150, 18, 236, 69, 57, 2, 216,
229, 10, 248, 157, 119, 9, 209, 165, 150, 193, 244, 31, 149, 170, 130, 202, 108, 73, 174, 144,
205, 22, 104, 186, 172, 122, 166, 242, 180, 168, 202, 153, 178, 194, 55, 42, 203, 8, 207, 97,
201, 195, 128, 94, 110, 3, 40, 218, 76, 215, 106, 25, 237, 210, 211, 153, 76, 121, 139, 0,
34, 86, 154, 212, 24, 209, 254, 228, 217, 205, 69, 163, 145, 198, 1, 255, 201, 42, 217, 21,
1, 67, 47, 238, 21, 2, 135, 97, 124, 19, 98, 158, 105, 252, 114, 129, 205, 113, 101, 166,
62, 171, 73, 207, 113, 75, 206, 58, 117, 167, 79, 118, 234, 126, 100, 255, 129, 235, 97, 253,
254, 195, 155, 103, 191, 13, 233, 140, 126, 78, 50, 189, 249, 124, 140, 106, 199, 91, 164, 60,
2, 244, 178, 237, 114, 22, 236, 243, 1, 77, 240, 0, 16, 139, 103, 207, 153, 80, 91, 23,
159, 142, 212, 152, 10, 97, 3, 209, 188, 167, 13, 190, 155, 191, 171, 14, 213, 152, 1, 214,
229, 242, 214, 246, 125, 62, 197, 22, 142, 33, 46, 45, 175, 2, 198, 185, 99, 201, 138, 31,
112, 151, 222, 12, 86, 137, 26, 43, 33, 27, 1, 7, 13, 216, 253, 139, 22, 194, 161, 164,
227, 207, 210, 146, 210, 152, 75, 53, 97, 213, 85, 209, 108, 51, 221, 194, 188, 247, 237, 222,
19, 239, 229, 32, 199, 226, 171, 221, 164, 77, 129, 136, 28, 83, 26, 238, 235, 102, 36, 76,
59, 121, 30, 168, 172, 251, 106, 104, 243, 88, 70, 6, 71, 43, 38, 14, 13, 210, 235, 178,
31, 108, 58, 59, 192, 84, 42, 171, 186, 78, 248, 246, 199, 22, 158, 115, 17, 8, 219, 4,
96, 34, 10, 167, 77, 49, 181, 91, 3, 160, 13, 34, 13, 71, 93, 205, 155, 135, 120, 86,
213, 112, 76, 156, 134, 234, 15, 152, 242, 235, 156, 83, 13, 167, 250, 90, 216, 176, 181, 219,
80, 194, 253, 93, 9, 90, 42, 165, 226, 163, 251, 183, 19, 71, 84, 154, 49, 99, 50, 35,
78, 206, 118, 91, 117, 113, 182, 77, 33, 107, 40, 113, 46, 37, 207, 55, 128, 249, 220, 98,
156, 215, 25, 176, 30, 109, 74, 79, 209, 124, 115, 31, 74, 233, 123, 192, 90, 49, 13, 123,
156, 54, 237, 202, 91, 188, 2, 219, 181, 222, 61, 82, 182, 87, 2, 212, 196, 76, 36, 149,
200, 151, 181, 18, 128, 48, 210, 219, 97, 224, 86, 253, 22, 67, 200, 113, 255, 202, 77, 181,
168, 138, 7, 94, 225, 9, 51, 166, 85, 87, 59, 29, 238, 240, 47, 110, 32, 2, 73, 129,
226, 160, 127, 248, 227, 71, 105, 227, 17, 182, 152, 185, 65, 159, 24, 34, 168, 75, 200, 253,
162, 4, 26, 144, 244, 73, 254, 21, 75, 72, 150, 45, 232, 21, 37, 203, 92, 143, 174, 109,
69, 70, 39, 134, 229, 63, 169, 141, 138, 113, 138, 44, 117, 164, 188, 106, 238, 186, 127, 57,
2, 21, 103, 234, 43, 140, 182, 135, 27, 100, 245, 97, 171, 28, 231, 144, 91, 144, 30, 229,
2, 168, 17, 119, 77, 205, 225, 59, 135, 96, 116, 138, 118, 219, 116, 161, 104, 42, 40, 131,
143, 29, 228, 58, 57, 204, 202, 148, 92, 232, 121, 94, 145, 138, 214, 222, 87, 183, 25, 223,
24, 141, 105, 142, 105, 221, 47, 209, 8, 87, 84, 151, 117, 57, 209, 174, 5, 155, 67, 97,
132, 188, 192, 21, 71, 150, 243, 158, 77, 12, 125, 101, 153, 230, 243, 2, 196, 34, 211, 204,
122, 40, 99, 239, 97, 52, 157, 102, 207, 224, 199, 83, 157, 135, 104, 228, 29, 91, 130, 107,
103, 0, 208, 1, 230, 196, 3, 170, 230, 215, 118, 96, 255, 217, 79, 96, 13, 237, 198, 221,
205, 141, 48, 106, 21, 153, 78, 50, 244, 209, 157, 92, 209, 110, 93, 183, 50, 96, 98, 24,
55, 216, 121, 54, 178, 200, 150, 191, 181, 92, 156, 131, 234, 205, 237, 255, 102, 60, 49, 90,
13, 207, 182, 222, 61, 19, 149, 111, 116, 247, 135, 171, 208, 0, 226, 130, 201, 120, 65, 126,
213, 222, 1, 191, 171, 239, 190, 17, 43, 239, 107, 56, 190, 34, 22, 251, 53, 171, 106, 169,
163, 242, 85, 115, 242, 55, 245, 187, 175, 54, 58, 132, 20, 59, 67, 191, 42, 1, 208, 85,
241, 60, 141, 175, 94, 163, 171, 147, 79, 21, 61, 242, 7, 146, 101, 250, 201, 90, 181, 120,
144, 239, 253, 165, 43, 64, 100, 85, 66, 53, 171, 51, 113, 56, 226, 207, 220, 141, 98, 43,
163, 159, 29, 170, 49, 130, 164, 250, 220, 90, 115, 108, 73, 112, 17, 116, 176, 118, 202, 242,
171, 117, 37, 28, 173, 8, 235, 137, 149, 77, 180, 56, 237, 209, 227, 30, 83, 135, 25, 47,
225, 140, 156, 43, 252, 173, 159, 172, 35, 105, 159, 206, 222, 196, 234, 140, 204, 213, 21, 98,
35, 202, 154, 16, 155, 125, 46, 239, 5, 71, 30, 230, 211, 186, 17, 207, 104, 177, 124, 139,
26, 27, 90, 249, 223, 68, 133, 172, 26, 154, 14, 61, 100, 168, 77, 0, 38, 123, 239, 43,
195, 13, 17, 150, 200, 35, 102, 48, 212, 226, 187, 238, 253, 21, 231, 220, 90, 108, 136, 116,
7, 150, 177, 107, 63, 254, 107, 101, 121, 90, 144, 60, 104, 161, 211, 48, 196, 57, 96, 152,
27, 27, 135, 24, 49, 110, 244, 139, 219, 125, 255, 226, 19, 176, 77, 82, 174, 185, 183, 39,
19, 71, 100, 123, 233, 55, 171, 173, 112, 11, 70, 139, 39, 205, 163, 88, 59, 151, 227, 22,
20, 226, 248, 40, 146, 70, 122, 64, 255, 50, 103, 18, 121, 203, 142, 98, 2, 57, 16, 114,
69, 86, 253, 108, 35, 160, 196, 94, 56, 167, 117, 76, 137, 109, 116, 27, 179, 239, 91, 178,
33, 194, 197, 154, 142, 83, 253, 144, 140, 13, 3, 209, 99, 0, 61, 134, 161, 1, 228, 217,
168, 89, 37, 49, 199, 154, 76, 122, 137, 167, 45, 170, 106, 242, 68, 248, 69, 65, 136, 209,
78, 139, 163, 177, 140, 224, 55, 45, 226, 28, 6, 138, 117, 43, 188, 60, 197, 8, 183, 78,
176, 228, 248, 26, 214, 61, 18, 27, 126, 154, 236, 205, 38, 143, 126, 178, 112, 182, 223, 82,
210, 229, 220, 71, 16, 152, 132, 214, 161, 59, 36, 81, 31, 29, 107, 245, 90, 125, 16, 216,
23, 252, 165, 61, 140, 36, 239, 252, 218, 206, 78, 172, 179, 42, 243, 196, 195, 119, 154, 100,
178, 190, 181, 209, 219, 32, 198, 53, 157, 214, 14, 180, 211, 179, 242, 95, 215, 225, 91, 177,
176, 169, 93, 99, 211, 81, 39, 150, 200, 193, 250, 123, 128, 175, 76, 91, 207, 19, 145, 108,
233, 159, 33, 188, 82, 19, 27, 42, 244, 118, 219, 164, 31, 57, 8, 243, 138, 47, 137, 82,
241, 132, 205, 113, 51, 26, 204, 3, 45, 93, 111, 22, 252, 144, 211, 79, 163, 238, 121, 152,
101, 84, 60, 132, 141, 68, 119, 23, 116, 1, 106, 101, 133, 55, 214, 184, 81, 162, 187, 126,
0, 43, 149, 252, 187, 104, 75, 95, 86, 196, 247, 187, 25, 51, 64, 166, 120, 183, 190, 236,
184, 40, 81, 61, 95, 39, 246, 177, 201, 177, 47, 201, 220, 196, 198, 152, 44, 17, 247, 131,
214, 238, 62, 239, 33, 126, 149, 153, 54, 83, 133, 238, 123, 214, 44, 219, 253, 34, 140, 199,
211, 187, 144, 176, 128, 86, 72, 172, 104, 63, 47, 62, 45, 110, 45, 78, 236, 194, 232, 34,
22, 109, 17, 145, 68, 61, 108, 65, 95, 248, 8, 50, 180, 153, 226, 52, 239, 42, 224, 87,
105, 16, 149, 150, 126, 194, 229, 106, 133, 205, 141, 155, 58, 158, 44, 126, 219, 153, 192, 58,
145, 200, 108, 69, 97, 79, 121, 81, 121, 90, 168, 227, 106, 62, 121, 232, 0, 94, 82, 133,
43, 223, 32, 102, 125, 77, 228, 88, 230, 164, 146, 119, 109, 255, 189, 206, 78, 54, 31, 199,
144, 200, 170, 250, 6, 36, 226, 6, 130, 53, 140, 174, 20, 172, 20, 146, 249, 248, 234, 223,
157, 125, 87, 10, 124, 20, 216, 202, 74, 248, 145, 219, 192, 60, 213, 198, 96, 184, 204, 226,
237, 88, 144, 1, 5, 164, 147, 254, 157, 126, 222, 58, 251, 53, 68, 119, 73, 28, 65, 147,
20, 210, 110, 212, 14, 68, 154, 110, 252, 103, 81, 233, 191, 225, 234, 196, 134, 126, 195, 35,
252, 161, 93, 247, 214, 161, 110, 31, 189, 175, 178, 210, 129, 33, 166, 144, 101, 65, 254, 97,
168, 79, 74, 103, 49, 52, 44, 183, 178, 239, 218, 174, 144, 55, 165, 102, 216, 19, 133, 149,
194, 55, 103, 68, 88, 14, 212, 189, 79, 210, 30, 247, 34, 104, 94, 83, 157, 138, 10, 79,
121, 228, 254, 9, 27, 163, 111, 243, 183, 244, 136, 121, 44, 240, 189, 132, 254, 145, 66, 77,
100, 96, 68, 134, 201, 162, 217, 102, 45, 227, 181, 166, 199, 179, 176, 226, 87, 31, 213, 14,
20, 93, 135, 64, 77, 69, 196, 75, 214, 6, 152, 58, 103, 220, 192, 48, 127, 153, 150, 172,
124, 75, 82, 67, 255, 2, 37, 86, 34, 250, 100, 54, 88, 235, 118, 165, 48, 58, 241, 7,
65, 137, 65, 168, 102, 2, 216, 229, 155, 110, 145, 24, 185, 227, 91, 184, 230, 129, 14, 8,
123, 114, 62, 211, 94, 180, 121, 142, 238, 106, 149, 47, 243, 215, 215, 89, 217, 175, 62, 116,
29, 207, 140, 215, 179, 232, 143, 153, 105, 158, 161, 228, 16, 223, 184, 110, 147, 49, 253, 129,
155, 146, 177, 142, 105, 136, 232, 66, 56, 38, 183, 85, 246, 67, 44, 169, 43, 188, 66, 148,
90, 227, 121, 106, 194, 49, 217, 85, 98, 214, 214, 253, 104, 135, 139, 210, 16, 115, 20, 72,
154, 203, 157, 144, 15, 202, 57, 58, 134, 123, 207, 224, 94, 72, 74, 32, 121, 35, 117, 219,
249, 75, 216, 98, 211, 99, 52, 227, 215, 72, 43, 113, 20, 200, 1, 35, 146, 58, 93, 24,
181, 44, 248, 19, 116, 67, 51, 237, 102, 168, 200, 96, 243, 160, 194, 198, 4, 246, 169, 219,
62, 212, 76, 82, 157, 77, 117, 47, 135, 211, 72, 60, 255, 64, 79, 116, 131, 130, 97, 234,
42, 42, 74, 29, 202, 12, 228, 206, 2, 141, 169, 64, 98, 245, 147, 255, 66, 8, 46, 201,
219, 118, 5, 219, 183, 84, 79, 58, 214, 176, 36, 0, 218, 110, 30, 165, 122, 2, 115, 124,
143, 29, 189, 241, 18, 80, 240, 85, 88, 31, 30, 52, 149, 36, 15, 76, 120, 94, 135, 79,
14, 171, 79, 233, 26, 109, 142, 148, 111, 1, 17, 255, 30, 206, 240, 49, 30, 225, 134, 118,
0, 164, 170, 149, 200, 185, 226, 65, 23, 105, 144, 38, 20, 223, 15, 46, 77, 157, 195, 188,
158, 212, 187, 189, 162, 172, 238, 192, 141, 116, 54, 141, 24, 225, 34, 225, 154, 4, 34, 178,
109, 178, 216, 130, 145, 231, 176, 222, 132, 115, 155, 34, 71, 86, 223, 233, 2, 205, 169, 143,
65, 224, 28, 90, 193, 63, 59, 91, 67, 93, 13, 177, 15, 229, 51, 160, 204, 227, 127, 80,
87, 26, 115, 158, 112, 82, 136, 115, 32, 49, 2, 97, 17, 31, 187, 210, 94, 246, 46, 161,
83, 59, 82, 98, 33, 133, 3, 237, 105, 130, 62, 192, 156, 177, 94, 12, 3, 230, 127, 35,
24, 130, 133, 41, 161, 64, 252, 255, 55, 42, 160, 138, 101, 243, 237, 134, 120, 240, 116, 225,
114, 178, 161, 14, 99, 0, 26, 102, 230, 154, 138, 254, 28, 15, 40, 189, 79, 36, 188, 134,
78, 92, 17, 179, 79, 254, 58, 200, 238, 174, 169, 96, 96, 75, 110, 195, 75, 136, 41, 49,
34, 179, 48, 62, 194, 88, 251, 18, 124, 183, 152, 202, 20, 169, 125, 99, 167, 183, 43, 149,
101, 213, 245, 197, 32, 99, 136, 107, 236, 178, 156, 14, 101, 204, 77, 40, 36, 72, 58, 160,
0, 210, 106, 20, 124, 232, 119, 35, 159, 163, 185, 5, 120, 174, 202, 152, 18, 83, 3, 254,
5, 159, 12, 106, 108, 89, 146, 144, 162, 204, 49, 162, 159, 155, 182, 27, 131, 45, 62, 35,
208, 247, 40, 72, 166, 242, 224, 184, 69, 227, 182, 74, 131, 194, 181, 239, 28, 71, 127, 190,
20, 176, 96, 179, 76, 22, 206, 207, 67, 12, 242, 20, 4, 26, 92, 170, 13, 61, 98, 82,
32, 24, 157, 163, 218, 82, 146, 246, 153, 18, 180, 173, 194, 20, 96, 14, 42, 46, 222, 110,
59, 208, 130, 63, 235, 222, 233, 248, 27, 75, 74, 60, 99, 231, 223, 61, 57, 114, 52, 211,
132, 232, 128, 70, 253, 225, 85, 39, 15, 51, 149, 74, 3, 23, 137, 238, 246, 114, 230, 17,
189, 49, 77, 32, 24, 45, 94, 82, 159, 146, 37, 35, 122, 165, 105, 119, 134, 190, 159, 150,
241, 52, 224, 245, 76, 106, 227, 66, 220, 202, 83, 154, 251, 161, 186, 19, 206, 24, 101, 109,
170, 138, 144, 37, 48, 249, 156, 182, 184, 59, 76, 169, 112, 45, 158, 188, 151, 130, 254, 115,
76, 81, 13, 71, 242, 200, 90, 192, 224, 192, 45, 139, 74, 189, 176, 122, 183, 76, 49, 111,
136, 125, 24, 248, 170, 183, 180, 65, 57, 178, 181, 133, 3, 194, 204, 246, 138, 38, 182, 107,
230, 228, 246, 49, 161, 166, 171, 88, 242, 220, 199, 122, 90, 224, 114, 4, 151, 38, 70, 208,
216, 251, 85, 220, 189, 33, 210, 72, 71, 136, 179, 46, 108, 169, 95, 14, 79, 10, 102, 65,
231, 46, 188, 65, 14, 46, 69, 165, 85, 139, 117, 45, 134, 202, 9, 68, 235, 219, 140, 50,
100, 63, 96, 208, 232, 191, 222, 55, 202, 69, 120, 177, 115, 52, 242, 129, 99, 55, 38, 184,
195, 155, 229, 73, 101, 239, 141, 80, 202, 25, 130, 46, 88, 227, 255, 64, 162, 221, 119, 108,
34, 240, 29, 149, 36, 15, 22, 135, 71, 60, 63, 10, 215, 37, 83, 60, 20, 225, 140, 222,
250, 15, 13, 83, 242, 12, 147, 148, 233, 11, 1, 12, 251, 30, 161, 31, 46, 184, 167, 117,
244, 230, 127, 204, 11, 210, 8, 31, 179, 149, 254, 174, 164, 11, 1, 150, 23, 148, 42, 0,
159, 43, 12, 154, 74, 174, 186, 120, 102, 97, 237, 90, 71, 108, 38, 83, 62, 47, 114, 242,
196, 112, 160, 104, 123, 161, 254, 146, 53, 40, 147, 213, 84, 159, 111, 158, 77, 41, 22, 179,
138, 3, 14, 210, 111, 52, 37, 173, 99, 151, 159, 39, 8, 63, 143, 131, 224, 141, 22, 22,
182, 169, 235, 10, 72, 90, 168, 150, 132, 190, 73, 14, 193, 87, 224, 48, 140, 5, 221, 239,
157, 125, 23, 165, 188, 166, 40, 157, 52, 62, 179, 234, 231, 158, 244, 48, 248, 156, 198, 124,
90, 15, 139, 27, 103, 107, 75, 243, 113, 40, 226, 14, 165, 249, 179, 98, 160, 219, 255, 212,
26, 178, 190, 1, 80, 178, 49, 72, 78, 247, 197, 168, 7, 80, 195, 110, 187, 14, 97, 44,
54, 67, 58, 220, 61, 237, 62, 221, 201, 61, 177, 227, 239, 111, 228, 63, 33, 22, 135, 111,
13, 76, 23, 20, 156, 218, 130, 88, 232, 227, 132, 30, 39, 191, 250, 100, 172, 56, 65, 117,
117, 242, 88, 100, 97, 61, 163, 130, 83, 43, 241, 96, 119, 8, 117, 20, 226, 247, 108, 202,
219, 240, 232, 2, 175, 227, 102, 92, 27, 167, 209, 145, 153, 42, 245, 250, 103, 153, 124, 186,
196, 109, 26, 59, 117, 143, 79, 87, 135, 187, 33, 98, 172, 9, 100, 91, 236, 202, 183, 8,
113, 137, 153, 10, 179, 142, 4, 26, 39, 128, 212, 235, 237, 238, 39, 98, 126, 118, 185, 5,
50, 218, 103, 222, 227, 203, 57, 208, 149, 241, 216, 6, 122, 113, 16, 45, 255, 20, 71, 39,
148, 27, 18, 129, 9, 57, 227, 135, 176, 156, 140, 226, 118, 243, 192, 89, 190, 249, 41, 83,
234, 1, 89, 100, 114, 105, 145, 114, 125, 216, 153, 17, 243, 171, 146, 253, 229, 117, 132, 149,
17, 17, 119, 135, 4, 55, 225, 195, 48, 10, 22, 27, 12, 112, 127, 126, 217, 17, 240, 87,
233, 137, 104, 221, 53, 251, 218, 26, 112, 94, 175, 130, 111, 38, 9, 116, 93, 234, 55, 141,
245, 77, 168, 1, 189, 40, 127, 151, 57, 112, 238, 34, 249, 86, 255, 46, 81, 217, 72, 194,
56, 247, 68, 167, 29, 77, 27, 122, 56, 82, 8, 45, 160, 176, 46, 93, 216, 173, 244, 17,
29, 226, 52, 23, 57, 51, 69, 138, 13, 142, 76, 69, 133, 144, 236, 163, 222, 8, 29, 22,
90, 37, 67, 250, 214, 113, 88, 174, 30, 76, 192, 60, 47, 244, 83, 104, 39, 152, 242, 52,
38, 63, 121, 172, 207, 102, 79, 173, 110, 108, 195, 200, 146, 6, 195, 104, 119, 27, 22, 150,
103, 214, 210, 150, 202, 37, 254, 242, 189, 241, 38, 228, 48, 160, 144, 255, 6, 223, 173, 116,
75, 112, 60, 221, 119, 255, 69, 238, 26, 92, 132, 130, 50, 86, 24, 253, 123, 23, 239, 57,
8, 21, 29, 56, 181, 173, 55, 187, 140, 228, 47, 215, 85, 108, 181, 204, 107, 250, 186, 134,
86, 63, 8, 137, 149, 32, 134, 17, 55, 117, 74, 63, 138, 103, 119, 64, 20, 175, 251, 160,
147, 43, 119, 232, 151, 44, 180, 2, 39, 111, 136, 125, 174, 144, 6, 67, 177, 140, 84, 232,
1, 158, 40, 140, 5, 159, 204, 25, 78, 199, 185, 226, 242, 49, 202, 137, 93, 127, 140, 132,
238, 20, 2, 156, 165, 8, 223, 86, 149, 52, 62, 150, 210, 102, 34, 216, 6, 238, 241, 84,
182, 171, 54, 168, 220, 1, 50, 57, 128, 190, 190, 110, 210, 192, 10, 119, 200, 233, 205, 93,
29, 12, 244, 240, 114, 22, 200, 120, 5, 185, 205, 187, 100, 3, 99, 64, 4, 149, 122, 132,
83, 56, 242, 38, 248, 252, 157, 192, 230, 107, 30, 3, 119, 18, 243, 233, 40, 187, 98, 45,
117, 47, 232, 217, 50, 76, 26, 55, 225, 148, 187, 53, 204, 174, 91, 196, 170, 248, 132, 144,
99, 162, 148, 218, 180, 135, 196, 221, 67, 38, 10, 184, 85, 243, 145, 135, 63, 171, 190, 32,
63, 122, 85, 11, 40, 176, 207, 210, 169, 84, 99, 12, 246, 247, 231, 171, 126, 171, 136, 193,
209, 146, 121, 38, 133, 11, 173, 196, 182, 108, 229, 246, 230, 58, 1, 14, 235, 209, 224, 148,
37, 67, 160, 27, 58, 135, 198, 185, 50, 78, 122, 3, 225, 244, 41, 102, 255, 215, 43, 181,
67, 16, 171, 41, 74, 173, 55, 53, 126, 23, 201, 163, 91, 106, 190, 149, 241, 133, 78, 36,
211, 201, 39, 180, 189, 81, 27, 188, 40, 70, 113, 107, 86, 29, 148, 160, 202, 203, 214, 72,
226, 159, 235, 61, 9, 169, 211, 251, 46, 33, 31, 2, 234, 70, 182, 167, 151, 209, 99, 191,
23, 213, 42, 109, 242, 190, 14, 188, 137, 228, 4, 108, 131, 239, 169, 141, 152, 124, 136, 199,
158, 167, 201, 136, 237, 127, 48, 133, 81, 147, 68, 104, 104, 111, 214, 90, 45, 228, 22, 183,
200, 26, 35, 75, 9, 204, 216, 162, 73, 96, 105, 231, 7, 50, 111, 245, 177, 159, 122, 2,
51, 190, 107, 155, 45, 65, 246, 91, 37, 12, 18, 237, 39, 53, 56, 48, 1, 17, 210, 74,
113, 59, 49, 121, 109, 161, 110, 31, 64, 232, 33, 115, 166, 140, 15, 212, 205, 5, 47, 242,
17, 65, 223, 56, 118, 24, 105, 119, 41, 59, 194, 154, 119, 243, 19, 228, 148, 129, 3, 213,
105, 37, 72, 16, 177, 87, 228, 127, 92, 19, 113, 110, 84, 81, 166, 202, 105, 15, 65, 146,
75, 3, 44, 194, 247, 64, 166, 139, 193, 170, 96, 43, 207, 169, 59, 128, 0, 31, 255, 93,
50, 113, 203, 134, 194, 113, 80, 43, 129, 145, 189, 204, 149, 233, 142, 140, 41, 52, 23, 235,
222, 120, 22, 173, 33, 81, 46, 33, 112, 45, 126, 162, 158, 73, 40, 96, 187, 120, 139, 60,
9, 72, 8, 158, 50, 150, 42, 91, 202, 66, 70, 169, 186, 92, 86, 219, 173, 132, 252, 29,
178, 123, 191, 80, 196, 231, 177, 127, 95, 60, 187, 105, 133, 195, 7, 183, 89, 50, 18, 36,
116, 89, 205, 46, 181, 35, 9, 99, 168, 5, 128, 90, 128, 64, 170, 69, 39, 91, 196, 135,
152, 128, 240, 29, 67, 247, 212, 157, 41, 230, 193, 157, 63, 142, 203, 245, 177, 212, 88, 89,
218, 216, 179, 90, 24, 94, 159, 64, 185, 100, 199, 81, 228, 183, 110, 39, 174, 66, 196, 216,
41, 133, 117, 104, 19, 65, 93, 197, 21, 181, 30, 239, 142, 210, 74, 166, 48, 233, 230, 233,
77, 173, 59, 49, 100, 169, 89, 19, 236, 29, 235, 21, 163, 96, 125, 182, 161, 219, 123, 183,
144, 154, 166, 30, 108, 240, 197, 156, 218, 171, 133, 39, 89, 192, 89, 189, 106, 178, 208, 86,
207, 187, 107, 114, 28, 232, 41, 189, 195, 164, 116, 84, 62, 27, 114, 170, 11, 55, 70, 229,
227, 204, 13, 60, 140, 102, 249, 246, 24, 202, 76, 231, 133, 183, 90, 161, 160, 131, 95, 99,
39, 211, 183, 102, 238, 42, 16, 250, 97, 87, 223, 68, 35, 236, 128, 175, 82, 122, 166, 106,
68, 242, 82, 201, 170, 172, 107, 74, 47, 202, 173, 86, 157, 101, 188, 140, 143, 205, 134, 240,
36, 101, 53, 71, 82, 181, 246, 164, 47, 156, 15, 115, 143, 97, 61, 57, 13, 168, 131, 60,
114, 48, 146, 15, 149, 79, 155, 36, 28, 33, 21, 64, 135, 74, 135, 217, 255, 125, 125, 47,
26, 140, 162, 169, 237, 223, 226, 250, 135, 101, 54, 249, 149, 201, 9, 43, 24, 164, 79, 52,
198, 100, 116, 77, 174, 251, 38, 174, 120, 163, 221, 146, 48, 127, 59, 29, 95, 29, 24, 230,
130, 78, 224, 24, 23, 233, 67, 47, 141, 146, 99, 83, 247, 215, 160, 165, 210, 198, 83, 75,
106, 48, 221, 154, 176, 25, 183, 15, 54, 207, 245, 185, 30, 213, 209, 53, 190, 20, 101, 76,
166, 200, 159, 157, 160, 64, 67, 114, 6, 150, 189, 112, 199, 155, 10, 119, 180, 194, 134, 234,
145, 123, 163, 175, 81, 116, 229, 15, 136, 74, 91, 47, 18, 251, 204, 178, 59, 15, 37, 65,
166, 226, 178, 109, 125, 188, 228, 49, 126, 106, 28, 16, 229, 191, 191, 54, 52, 164, 70, 188,
238, 161, 235, 1, 156, 184, 179, 215, 199, 216, 25, 109, 187, 203, 218, 56, 135, 190, 106, 6,
40, 134, 22, 14, 69, 213, 68, 121, 122, 138, 54, 104, 44, 33, 105, 200, 217, 29, 160, 161,
245, 185, 14, 176, 132, 233, 233, 11, 167, 83, 17, 208, 217, 39, 222, 30, 253, 34, 152, 119,
173, 206, 223, 217, 239, 73, 161, 201, 102, 65, 106, 91, 250, 120, 12, 126, 97, 245, 138, 9,
72, 155, 217, 33, 195, 183, 63, 192, 217, 215, 55, 134, 165, 22, 95, 149, 95, 1, 94, 197,
66, 200, 33, 61, 64, 45, 187, 162, 34, 69, 171, 106, 225, 132, 139, 164, 59, 202, 100, 20,
162, 155, 155, 71, 177, 250, 220, 17, 251, 58, 214, 62, 2, 247, 123, 67, 36, 54, 229, 70,
124, 144, 176, 93, 20, 59, 1, 79, 6, 101, 99, 168, 0, 254, 239, 177, 249, 204, 194, 244,
6, 153, 50, 9, 144, 173, 76, 181, 228, 49, 251, 96, 193, 172, 189, 213, 231, 190, 36, 237,
35, 135, 149, 35, 134, 133, 212, 127, 81, 151, 115, 87, 48, 166, 96, 192, 83, 172, 117, 55,
221, 113, 151, 158, 29, 84, 115, 4, 18, 151, 242, 53, 31, 135, 88, 165, 12, 45, 36, 93,
196, 151, 181, 244, 61, 21, 180, 145, 194, 42, 200, 159, 155, 96, 62, 184, 180, 177, 188, 199,
73, 174, 252, 104, 54, 85, 13, 66, 130, 49, 160, 70, 200, 85, 58, 6, 106, 238, 151, 44,
24, 95, 204, 179, 191, 10, 107, 116, 187, 40, 59, 4, 214, 55, 108, 12, 140, 121, 79, 14,
170, 239, 84, 115, 68, 142, 121, 174, 125, 16, 219, 149, 111, 167, 73, 47, 177, 180, 163, 108,
220, 222, 113, 179, 21, 221, 191, 162, 87, 14, 176, 1, 253, 5, 116, 65, 147, 237, 240, 16,
253, 203, 166, 109, 114, 239, 156, 35, 163, 63, 143, 128, 29, 0, 51, 50, 222, 242, 212, 53,
1, 133, 54, 254, 138, 171, 64, 29, 152, 48, 46, 150, 251, 212, 3, 109, 195, 159, 144, 102,
222, 31, 230, 251, 32, 25, 45, 254, 12, 2, 51, 13, 135, 105, 11, 17, 20, 75, 46, 173,
123, 92, 67, 118, 48, 70, 227, 243, 229, 115, 90, 195, 147, 64, 190, 179, 90, 235, 177, 102,
237, 228, 115, 116, 77, 126, 133, 98, 202, 180, 15, 69, 16, 82, 188, 65, 152, 159, 52, 125,
19, 142, 64, 166, 207, 254, 89, 41, 233, 10, 143, 215, 238, 2, 75, 59, 128, 209, 157, 74,
133, 172, 144, 149, 254, 76, 214, 150, 235, 11, 19, 254, 153, 83, 164, 104, 81, 253, 145, 59,
7, 32, 18, 245, 34, 93, 49, 163, 46, 206, 237, 179, 123, 125, 73, 121, 201, 31, 16, 181,
42, 35, 179, 196, 119, 88, 44, 200, 85, 190, 3, 93, 222, 21, 82, 1, 115, 131, 164, 161,
82, 145, 85, 205, 15, 158, 70, 216, 189, 86, 141, 232, 122, 65, 172, 241, 153, 216, 185, 238,
150, 189, 75, 117, 210, 158, 118, 69, 33, 26, 231, 115, 171, 60, 64, 186, 218, 135, 147, 151,
221, 32, 127, 87, 97, 43, 72, 250, 4, 2, 233, 154, 191, 52, 15, 145, 210, 133, 215, 244,
159, 190, 103, 75, 250, 168, 5, 212, 47, 152, 107, 12, 185, 235, 100, 26, 22, 172, 21, 26,
174, 254, 181, 109, 50, 196, 255, 5, 74, 214, 249, 233, 148, 96, 52, 142, 8, 58, 98, 55,
210, 205, 68, 139, 184, 168, 166, 207, 84, 187, 233, 3, 185, 158, 112, 235, 99, 111, 240, 173,
69, 233, 150, 217, 74, 203, 103, 82, 5, 201, 138, 215, 151, 206, 99, 79, 118, 9, 30, 202,
196, 8, 205, 125, 166, 62, 104, 9, 173, 89, 182, 243, 66, 77, 204, 140, 24, 52, 223, 29,
253, 105, 244, 148, 55, 87, 228, 173, 96, 2, 119, 36, 10, 69, 161, 177, 131, 10, 186, 48,
99, 113, 35, 165, 190, 240, 50, 214, 36, 17, 243, 33, 122, 231, 182, 177, 63, 154, 94, 159,
156, 213, 196, 167, 26, 101, 88, 157, 111, 18, 206, 210, 131, 241, 120, 65, 225, 170, 23, 5,
187, 10, 39, 53, 242, 221, 230, 49, 119, 68, 208, 19, 25, 148, 186, 52, 250, 18, 209, 105,
37, 159, 60, 168, 145, 180, 234, 114, 94, 1, 120, 25, 12, 159, 78, 254, 124, 52, 47, 243,
120, 255, 6, 145, 148, 193, 197, 142, 211, 151, 247, 248, 54, 51, 161, 199, 231, 139, 58, 69,
140, 178, 94, 152, 81, 172, 150, 205, 224, 197, 192, 88, 197, 198, 234, 89, 135, 175, 231, 91,
70, 222, 83, 125, 18, 244, 68, 249, 127, 126, 63, 12, 48, 157, 164, 129, 74, 59, 78, 42,
0, 14, 131, 197, 213, 109, 30, 92, 28, 5, 183, 99, 228, 11, 224, 246, 255, 36, 239, 127,
163, 46, 139, 211, 204, 47, 85, 22, 106, 163, 64, 107, 178, 195, 48, 135, 48, 79, 227, 77,
84, 155, 176, 56, 166, 144, 46, 165, 180, 30, 36, 87, 76, 175, 43, 24, 223, 128, 46, 73,
35, 111, 180, 213, 50, 229, 92, 99, 52, 64, 176, 136, 219, 96, 193, 129, 240, 239, 38, 164,
13, 75, 252, 90, 250, 39, 114, 217, 167, 161, 35, 202, 16, 215, 160, 66, 188, 252, 165, 240,
60, 85, 121, 23, 181, 58, 152, 165, 41, 191, 74, 55, 10, 70, 145, 4, 109, 3, 222, 20,
164, 1, 222, 180, 216, 126, 247, 149, 123, 156, 133, 183, 242, 254, 207, 167, 56, 103, 77, 98,
38, 151, 153, 48, 221, 42, 53, 74, 45, 19, 94, 210, 20, 60, 134, 236, 187, 125, 129, 54,
26, 7, 237, 12, 5, 188, 179, 62, 36, 0, 160, 74, 151, 57, 123, 116, 99, 176, 190, 144,
195, 28, 98, 215, 89, 233, 195, 20, 102, 69, 74, 128, 76, 55, 140, 81, 244, 64, 143, 24,
64, 47, 98, 216, 104, 221, 76, 203, 141, 11, 92, 80, 39, 190, 39, 128, 167, 235, 148, 14,
48, 222, 142, 124, 22, 27, 205, 10, 91, 93, 34, 155, 140, 132, 115, 245, 98, 192, 192, 239,
203, 28, 64, 242, 219, 103, 115, 130, 82, 7, 144, 130, 230, 31, 254, 252, 58, 204, 6, 149,
41, 40, 48, 181, 172, 164, 170, 14, 100, 107, 254, 47, 135, 62, 33, 98, 165, 148, 229, 248,
156, 117, 122, 130, 148, 121, 126, 206, 69, 132, 99, 110, 172, 148, 35, 88, 56, 206, 103, 156,
57, 101, 203, 192, 163, 236, 35, 72, 129, 8, 64, 29, 125, 187, 159, 18, 52, 29, 224, 121,
161, 68, 231, 77, 216, 10, 165, 16, 216, 12, 172, 17, 113, 119, 210, 20, 99, 245, 93, 228,
253, 157, 1, 122, 88, 160, 140, 140, 189, 109, 5, 94, 177, 236, 171, 137, 247, 81, 153, 207,
93, 69, 225, 207, 188, 179, 227, 31, 168, 64, 4, 165, 222, 5, 31, 54, 166, 172, 195, 99,
25, 200, 194, 202, 181, 109, 83, 172, 190, 236, 123, 28, 49, 92, 235, 237, 15, 206, 12, 183,
15, 16, 92, 237, 22, 124, 35, 188, 40, 230, 31, 65, 175, 225, 11, 100, 79, 94, 16, 13,
74, 139, 41, 123, 232, 20, 104, 247, 227, 116, 175, 242, 133, 11, 223, 155, 135, 2, 87, 175,
233, 118, 240, 152, 88, 251, 252, 167, 89, 12, 180, 163, 151, 222, 30, 127, 242, 134, 119, 213,
251, 38, 199, 128, 49, 166, 27, 185, 169, 114, 104, 146, 232, 89, 42, 64, 84, 38, 231, 174,
50, 156, 81, 201, 122, 112, 73, 108, 246, 192, 66, 241, 230, 9, 113, 23, 176, 140, 208, 89,
254, 57, 235, 231, 146, 21, 39, 230, 59, 15, 148, 109, 171, 230, 54, 37, 86, 127, 145, 76,
63, 211, 62, 37, 221, 175, 61, 141, 60, 13, 230, 58, 70, 209, 33, 216, 230, 73, 191, 33,
88, 83, 142, 3, 57, 196, 40, 143, 68, 185, 220, 131, 141, 26, 169, 106, 201, 230, 247, 5,
243, 221, 64, 58, 174, 97, 18, 148, 170, 209, 181, 2, 37, 67, 5, 94, 7, 45, 238, 75,
231, 202, 207, 116, 228, 120, 222, 173, 94, 213, 179, 81, 178, 243, 139, 96, 84, 158, 244, 255,
111, 169, 1, 148, 236, 7, 243, 243, 52, 225, 63, 27, 171, 14, 143, 143, 134, 109, 60, 228,
66, 239, 53, 244, 226, 193, 84, 55, 95, 72, 54, 206, 241, 55, 99, 221, 62, 86, 209, 115,
55, 16, 142, 226, 30, 30, 113, 164, 139, 173, 136, 206, 157, 189, 194, 127, 126, 23, 182, 221,
95, 236, 172, 81, 36, 15, 46, 98, 101, 255, 213, 156, 15, 100, 126, 45, 130, 239, 209, 13,
156, 89, 219, 57, 23, 158, 185, 149, 181, 111, 115, 20, 92, 31, 101, 128, 46, 148, 226, 147,
147, 184, 47, 163, 28, 173, 208, 158, 156, 162, 171, 56, 251, 135, 114, 18, 37, 43, 168, 218,
154, 27, 238, 246, 58, 84, 118, 104, 232, 89, 251, 123, 17, 42, 30, 45, 215, 239, 203, 115,
145, 118, 171, 140, 253, 29, 159, 34, 72, 71, 252, 227, 98, 235, 217, 156, 63, 80, 4, 39,
169, 255, 162, 186, 41, 193, 231, 0, 176, 178, 115, 65, 40, 30, 205, 38, 60, 108, 72, 132,
179, 69, 103, 21, 48, 65, 177, 111, 145, 181, 150, 58, 180, 56, 244, 221, 249, 219, 221, 169,
141, 80, 234, 181, 111, 184, 219, 171, 36, 36, 47, 216, 105, 151, 237, 153, 216, 159, 8, 105,
84, 158, 163, 9, 214, 151, 230, 208, 114, 196, 121, 255, 20, 100, 180, 131, 28, 144, 46, 64,
180, 94, 24, 29, 245, 6, 182, 205, 165, 190, 54, 249, 92, 217, 2, 50, 112, 233, 2, 226,
173, 124, 225, 193, 224, 149, 69, 252, 37, 115, 60, 217, 209, 85, 246, 198, 91, 172, 147, 0,
106, 201, 249, 198, 162, 252, 249, 18, 229, 251, 244, 146, 119, 213, 83, 87, 107, 152, 83, 144,
12, 144, 106, 221, 229, 96, 164, 64, 13, 55, 64, 119, 1, 57, 62, 163, 53, 55, 182, 26,
50, 170, 172, 170, 128, 0, 1, 235, 152, 85, 123, 164, 229, 229, 130, 202, 70, 38, 10, 83,
93, 74, 202, 94, 131, 8, 2, 185, 63, 184, 211, 114, 98, 128, 28, 226, 128, 29, 205, 24,
114, 73, 189, 87, 46, 63, 33, 116, 101, 43, 199, 194, 117, 146, 33, 249, 154, 35, 178, 218,
219, 133, 76, 61, 5, 104, 32, 133, 133, 237, 158, 248, 54, 91, 79, 101, 154, 113, 217, 255,
156, 161, 193, 18, 51, 226, 11, 205, 5, 189, 167, 224, 66, 243, 30, 72, 91, 62, 205, 225,
43, 107, 217, 98, 198, 40, 199, 96, 153, 160, 95, 54, 65, 33, 72, 116, 3, 83, 66, 9,
16, 233, 233, 82, 221, 7, 154, 56, 69, 104, 25, 113, 211, 242, 211, 154, 27, 154, 250, 180,
58, 90, 234, 124, 123, 50, 240, 126, 133, 50, 135, 149, 28, 113, 232, 249, 120, 130, 49, 190,
234, 75, 47, 190, 61, 2, 88, 88, 156, 82, 13, 214, 172, 247, 82, 39, 42, 67, 166, 175,
117, 45, 69, 145, 158, 45, 138, 23, 175, 188, 213, 154, 7, 4, 88, 68, 6, 176, 157, 162,
2, 170, 120, 175, 161, 203, 214, 203, 14, 124, 123, 131, 170, 192, 21, 72, 237, 159, 95, 156,
91, 52, 54, 98, 56, 142, 167, 62, 62, 68, 224, 65, 238, 89, 240, 143, 36, 198, 91, 50,
67, 214, 181, 237, 150, 202, 53, 131, 106, 149, 31, 197, 201, 86, 40, 2, 228, 207, 64, 35,
19, 33, 100, 1, 122, 84, 144, 158, 26, 235, 208, 93, 193, 133, 74, 87, 80, 128, 218, 186,
21, 250, 127, 222, 80, 167, 224, 52, 118, 33, 87, 137, 66, 187, 138, 188, 15, 27, 90, 42,
6, 42, 135, 200, 175, 210, 31, 255, 82, 250, 185, 103, 244, 57, 69, 68, 224, 38, 120, 87,
71, 208, 224, 137, 139, 107, 69, 155, 134, 159, 197, 140, 201, 76, 84, 120, 30, 116, 120, 112,
110, 49, 215, 98, 106, 29, 166, 75, 67, 30, 162, 138, 238, 130, 19, 122, 237, 88, 21, 115,
247, 218, 0, 192, 38, 84, 56, 69, 200, 176, 181, 54, 226, 141, 152, 76, 170, 62, 151, 237,
93, 57, 119, 75, 188, 138, 197, 169, 226, 218, 29, 217, 180, 29, 153, 219, 113, 209, 32, 58,
130, 213, 112, 100, 98, 9, 176, 12, 71, 72, 249, 164, 129, 112, 240, 61, 250, 181, 231, 220,
144, 4, 181, 68, 33, 78, 31, 146, 32, 63, 204, 162, 21, 61, 6, 119, 70, 182, 132, 141,
254, 125, 50, 128, 238, 34, 189, 232, 215, 164, 197, 103, 168, 122, 172, 201, 201, 203, 92, 233,
11, 40, 139, 32, 101, 145, 151, 171, 71, 27, 57, 70, 153, 107, 198, 135, 141, 131, 111, 100,
40, 52, 204, 208, 175, 120, 154, 120, 67, 246, 97, 78, 30, 236, 110, 132, 125, 6, 47, 196,
33, 104, 10, 186, 211, 208, 65, 96, 84, 177, 197, 124, 229, 145, 76, 148, 9, 230, 12, 76,
220, 109, 155, 251, 89, 9, 127, 214, 15, 174, 155, 49, 23, 165, 235, 234, 118, 45, 75, 202,
222, 16, 70, 195, 161, 146, 88, 170, 121, 100, 246, 85, 210, 145, 80, 43, 155, 207, 2, 170,
126, 157, 219, 149, 66, 199, 127, 184, 244, 202, 130, 210, 218, 200, 149, 123, 91, 237, 37, 212,
82, 28, 41, 36, 173, 122, 79, 72, 73, 81, 243, 199, 238, 206, 92, 49, 149, 220, 233, 137,
166, 108, 91, 129, 52, 241, 252, 143, 222, 34, 99, 48, 62, 141, 84, 235, 7, 164, 52, 80,
245, 39, 24, 228, 245, 116, 21, 139, 80, 254, 20, 247, 106, 112, 120, 159, 97, 116, 46, 63,
150, 146, 112, 212, 31, 196, 192, 38, 104, 244, 118, 94, 27, 142, 66, 16, 3, 87, 155, 83,
85, 176, 74, 192, 32, 194, 95, 129, 55, 141, 192, 205, 31, 48, 162, 62, 245, 98, 100, 93,
86, 219, 187, 113, 105, 253, 129, 108, 84, 29, 192, 170, 205, 10, 106, 237, 205, 201, 110, 4,
86, 46, 209, 118, 95, 115, 180, 84, 213, 25, 177, 43, 244, 109, 156, 93, 106, 30, 202, 191,
59, 138, 105, 8, 148, 211, 245, 97, 156, 99, 101, 242, 145, 55, 104, 240, 170, 29, 68, 128,
54, 246, 171, 42, 99, 72, 135, 205, 102, 81, 140, 161, 219, 245, 169, 112, 200, 158, 209, 100,
1, 55, 87, 146, 110, 191, 131, 24, 220, 199, 152, 18, 189, 68, 60, 32, 140, 196, 238, 242,
21, 122, 147, 241, 112, 60, 97, 56, 218, 50, 157, 219, 105, 244, 109, 215, 179, 240, 240, 144,
184, 136, 162, 117, 204, 223, 150, 88, 163, 132, 74, 184, 254, 221, 169, 110, 25, 10, 167, 243,
61, 68, 206, 166, 56, 60, 126, 235, 44, 110, 123, 228, 246, 30, 90, 195, 253, 240, 27, 160,
116, 102, 88, 114, 67, 2, 225, 93, 12, 136, 80, 73, 204, 31, 240, 4, 91, 110, 239, 135,
220, 107, 108, 210, 137, 198, 149, 134, 182, 177, 38, 42, 23, 126, 156, 90, 128, 125, 183, 141,
5, 8, 214, 209, 39, 198, 213, 130, 52, 197, 9, 16, 48, 117, 227, 185, 59, 120, 63, 241,
41, 101, 27, 64, 227, 184, 155, 100, 53, 82, 241, 59, 90, 199, 12, 129, 142, 226, 3, 194,
167, 13, 211, 215, 130, 182, 144, 190, 46, 207, 175, 88, 52, 203, 152, 23, 131, 51, 123, 184,
134, 108, 243, 224, 52, 0, 98, 194, 226, 101, 132, 137, 114, 87, 96, 245, 13, 240, 179, 60,
191, 98, 148, 243, 45, 44, 10, 176, 96, 134, 105, 230, 242, 92, 198, 38, 92, 40, 232, 62,
142, 109, 199, 0, 196, 39, 245, 210, 23, 168, 14, 214, 11, 162, 201, 56, 206, 212, 233, 46,
90, 82, 20, 76, 174, 219, 115, 11, 3, 91, 73, 145, 200, 17, 146, 141, 56, 135, 95, 80,
48, 109, 38, 59, 15, 240, 115, 221, 196, 92, 12, 30, 174, 32, 106, 93, 251, 221, 104, 255,
57, 177, 144, 1, 194, 34, 142, 251, 170, 237, 75, 218, 90, 113, 21, 105, 97, 136, 71, 37,
229, 83, 67, 147, 115, 174, 240, 111, 139, 88, 110, 196, 10, 254, 198, 204, 33, 84, 199, 203,
66, 18, 165, 156, 132, 186, 6, 229, 66, 77, 11, 39, 160, 78, 187, 19, 252, 171, 130, 136,
4, 240, 76, 14, 239, 18, 218, 16, 103, 162, 219, 169, 180, 128, 69, 56, 58, 75, 30, 124,
152, 41, 164, 56, 119, 95, 76, 116, 10, 206, 252, 14, 191, 72, 28, 174, 91, 247, 190, 194,
153, 153, 107, 77, 25, 176, 134, 83, 252, 164, 207, 148, 205, 115, 205, 68, 210, 25, 184, 221,
231, 180, 235, 166, 253, 8, 84, 88, 255, 18, 26, 152, 171, 133, 229, 196, 53, 107, 23, 49,
15, 231, 198, 220, 90, 147, 33, 45, 172, 217, 10, 147, 142, 245, 58, 139, 253, 142, 227, 252,
161, 253, 148, 76, 130, 122, 17, 183, 229, 40, 233, 245, 15, 175, 209, 106, 66, 242, 151, 238,
204, 161, 129, 90, 150, 187, 229, 148, 74, 200, 144, 235, 197, 37, 55, 71, 159, 72, 254, 132,
113, 231, 121, 128, 150, 75, 234, 216, 61, 129, 198, 9, 34, 72, 99, 185, 3, 72, 77, 77,
16, 221, 56, 213, 2, 112, 28, 161, 184, 27, 38, 41, 2, 159, 170, 153, 234, 148, 113, 40,
22, 56, 49, 56, 128, 149, 241, 131, 221, 62, 209, 238, 28, 9, 195, 30, 121, 224, 192, 50,
251, 230, 91, 253, 133, 5, 150, 112, 154, 8, 152, 176, 64, 201, 232, 192, 94, 218, 67, 60,
24, 20, 42, 52, 30, 237, 83, 151, 205, 19, 201, 200, 249, 37, 198, 126, 42, 92, 238, 196,
100, 134, 116, 164, 80, 93, 100, 174, 55, 168, 234, 79, 188, 20, 132, 218, 2, 215, 114, 207,
234, 59, 152, 227, 96, 94, 97, 139, 186, 80, 79, 31, 214, 196, 195, 38, 33, 40, 213, 88,
208, 191, 167, 140, 212, 43, 103, 214, 2, 217, 165, 236, 20, 61, 207, 117, 155, 49, 0, 86,
129, 79, 117, 87, 19, 56, 126, 52, 96, 83, 140, 48, 18, 52, 189, 230, 95, 36, 188, 98,
253, 98, 78, 17, 159, 30, 134, 59, 79, 134, 145, 208, 214, 6, 39, 233, 62, 165, 30, 159,
248, 170, 207, 11, 222, 140, 241, 62, 176, 174, 160, 173, 16, 238, 191, 175, 12, 69, 234, 91,
204, 123, 43, 162, 129, 83, 139, 192, 248, 169, 95, 241, 84, 46, 252, 50, 187, 237, 112, 107,
155, 16, 25, 171, 255, 216, 91, 11, 29, 69, 103, 233, 193, 146, 139, 66, 229, 23, 2, 222,
192, 97, 207, 20, 144, 203, 71, 75, 184, 183, 182, 84, 200, 207, 255, 199, 167, 90, 210, 197,
160, 57, 174, 97, 204, 58, 163, 177, 81, 166, 143, 17, 7, 94, 38, 151, 41, 109, 226, 226,
36, 153, 54, 236, 104, 53, 179, 16, 144, 134, 213, 48, 191, 131, 145, 139, 189, 52, 61, 14,
218, 204, 32, 226, 43, 70, 121, 84, 179, 92, 54, 215, 245, 108, 196, 93, 162, 119, 109, 50,
253, 66, 98, 189, 198, 243, 72, 131, 39, 133, 146, 2, 82, 178, 228, 125, 248, 93, 209, 171,
185, 8, 130, 174, 116, 70, 12, 22, 190, 121, 72, 187, 188, 170, 120, 130, 157, 193, 5, 197,
70, 151, 199, 152, 73, 171, 21, 65, 8, 231, 236, 194, 239, 111, 112, 99, 181, 124, 122, 115,
246, 194, 47, 178, 109, 167, 52, 10, 104, 57, 207, 175, 209, 150, 71, 26, 65, 93, 92, 74,
68, 72, 12, 51, 183, 124, 150, 109, 249, 16, 224, 239, 211, 15, 161, 64, 183, 213, 74, 31,
14, 26, 206, 223, 176, 22, 250, 242, 115, 86, 60, 183, 158, 72, 234, 86, 196, 128, 195, 189,
145, 163, 172, 100, 179, 77, 164, 106, 34, 238, 137, 49, 8, 88, 16, 185, 110, 10, 171, 225,
96, 231, 152, 255, 47, 130, 85, 243, 2, 24, 177, 147, 187, 93, 247, 110, 171, 155, 216, 205,
138, 98, 254, 146, 186, 15, 75, 40, 25, 246, 9, 122, 221, 161, 121, 12, 35, 206, 0, 37,
230, 177, 185, 161, 14, 176, 16, 185, 76, 232, 135, 214, 74, 133, 104, 4, 148, 180, 44, 174,
170, 53, 40, 136, 214, 161, 148, 249, 111, 148, 31, 85, 69, 216, 246, 84, 136, 6, 13, 212,
239, 148, 170, 57, 26, 19, 62, 174, 199, 106, 92, 113, 160, 132, 249, 118, 37, 142, 112, 148,
34, 143, 233, 104, 103, 224, 188, 239, 230, 201, 196, 213, 94, 110, 15, 120, 129, 77, 38, 72,
183, 131, 186, 87, 7, 179, 206, 45, 65, 62, 193, 100, 205, 171, 204, 52, 139, 136, 35, 113,
81, 231, 71, 175, 86, 86, 39, 215, 163, 78, 32, 90, 209, 218, 178, 216, 141, 128, 5, 207,
190, 199, 51, 139, 114, 255, 191, 253, 135, 226, 110, 216, 202, 181, 136, 32, 11, 175, 247, 174,
253, 23, 9, 206, 241, 187, 167, 127, 59, 172, 78, 249, 115, 129, 132, 229, 128, 67, 226, 7,
37, 81, 223, 239, 6, 103, 15, 18, 23, 7, 192, 20, 30, 201, 227, 16, 132, 138, 143, 191,
54, 221, 184, 170, 94, 60, 143, 222, 127, 114, 229, 165, 195, 196, 148, 201, 44, 164, 219, 67,
171, 156, 87, 201, 101, 58, 217, 234, 196, 104, 169, 251, 69, 98, 165, 163, 158, 52, 129, 30,
166, 102, 195, 105, 43, 87, 51, 87, 251, 14, 154, 166, 170, 241, 112, 16, 44, 73, 250, 240,
178, 163, 235, 247, 5, 144, 155, 164, 197, 28, 194, 107};
