/*  Inti-GConf: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  changeset.cc - GConfChangeSet C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "changeset.h"
#include "schema.h"
#include "value.h"
#include "internals.h"
#include <inti/glib/error.h>
#include <glib/gmessages.h>

using namespace Inti;

/*  GConf::ChangeSet
 */

GConf::ChangeSet::ChangeSet()
: cs_(gconf_change_set_new())
{
	set_referenced(true);
	gconf_change_set_set_user_data(gconf_change_set(), this, &destroy_notify);
}

GConf::ChangeSet::ChangeSet(GConfChangeSet *cs)
: cs_(cs)
{
}

GConf::ChangeSet::~ChangeSet()
{
	if (cs_)
		cs_ = 0;
}

void 
GConf::ChangeSet::destroy_notify(void *data)
{
	GConf::ChangeSet *change_set = static_cast<GConf::ChangeSet*>(data);
	if (change_set)
	{
		change_set->cs_ = 0;
		if (change_set->is_dynamic())
			delete change_set;
	}
}

void
GConf::ChangeSet::ref()
{
	gconf_change_set_ref(gconf_change_set());
}

void 
GConf::ChangeSet::unref()
{
	gconf_change_set_unref(gconf_change_set());
}

void 
GConf::ChangeSet::clear()
{
	gconf_change_set_clear(gconf_change_set());
}

unsigned int 
GConf::ChangeSet::size() const
{
	return gconf_change_set_size(gconf_change_set());
}

void 
GConf::ChangeSet::remove(const String& key)
{
	gconf_change_set_remove(gconf_change_set(), key.c_str());
}

namespace {

void foreach_slot_callback(GConfChangeSet*, const gchar* key, GConfValue* value, gpointer user_data)
{
	GConf::ChangeSet::ForeachSlot *slot = static_cast<GConf::ChangeSet::ForeachSlot*>(user_data);
	String tmp_key(key);
	GConf::Value tmp_value(value);
	slot->call(tmp_key, value ? &tmp_value : 0);
}

} // namespace

void 
GConf::ChangeSet::foreach(ForeachSlot *each)
{
	gconf_change_set_foreach(gconf_change_set(), &foreach_slot_callback, (void*)each);
}

bool 
GConf::ChangeSet::check_value(const String& key, Value *value)
{
	GConfValue *tmp_value = 0;
	bool result = gconf_change_set_check_value(gconf_change_set(), key.c_str(), &tmp_value);
	if (value)
		*value = Value(tmp_value);
	return result;
}

void 
GConf::ChangeSet::set(const String& key, const Value& value)
{
	gconf_change_set_set(gconf_change_set(), key.c_str(), value.gconf_value());
}

void 
GConf::ChangeSet::unset(const String& key)
{
	gconf_change_set_unset(gconf_change_set(), key.c_str());
}

void 
GConf::ChangeSet::set_int(const String& key, int value)
{
	gconf_change_set_set_int(gconf_change_set(), key.c_str(), value);
}

void 
GConf::ChangeSet::set_bool(const String& key, bool value)
{
	gconf_change_set_set_bool(gconf_change_set(), key.c_str(), value);
}
	
void 
GConf::ChangeSet::set_float(const String& key, double value)
{
	gconf_change_set_set_float(gconf_change_set(), key.c_str(), value);
}

void 
GConf::ChangeSet::set_string(const String& key, const String& value)
{
	gconf_change_set_set_string(gconf_change_set(), key.c_str(), value.c_str());
}

void 
GConf::ChangeSet::set_schema(const String& key, const Schema& value)
{
	gconf_change_set_set_schema(gconf_change_set(), key.c_str(), value.gconf_schema());
}

void 
GConf::ChangeSet::set_list(const String& key, const std::vector<int>& list)
{
	g_return_if_fail(!list.empty());
	GSList *tmp_list = vector_to_gslist(key, list);
	gconf_change_set_set_list(gconf_change_set(), key.c_str(), GCONF_VALUE_INT, tmp_list); 
 	g_slist_free(tmp_list);
}

void 
GConf::ChangeSet::set_list(const String& key, const std::vector<bool>& list)
{
	g_return_if_fail(!list.empty());
	GSList *tmp_list = vector_to_gslist(key, list);
	gconf_change_set_set_list(gconf_change_set(), key.c_str(), GCONF_VALUE_BOOL, tmp_list);  
 	g_slist_free(tmp_list);
}

void 
GConf::ChangeSet::set_list(const String& key, const std::vector<double>& list)
{
	g_return_if_fail(!list.empty());
	GSList *tmp_list = vector_to_gslist(key, list);
	gconf_change_set_set_list(gconf_change_set(), key.c_str(), GCONF_VALUE_FLOAT, tmp_list); 
 	g_slist_free(tmp_list);
}

void 
GConf::ChangeSet::set_list(const String& key, const std::vector<String>& list)
{
	g_return_if_fail(!list.empty());
	GSList *tmp_list = vector_to_gslist(key, list);
	gconf_change_set_set_list(gconf_change_set(), key.c_str(), GCONF_VALUE_STRING, tmp_list);    
	g_slist_free(tmp_list);
}

void 
GConf::ChangeSet::set_list(const String& key, const std::vector<Schema>& list)
{
	g_return_if_fail(!list.empty());
	GSList *tmp_list = vector_to_gslist(key, list);
	gconf_change_set_set_list(gconf_change_set(), key.c_str(), GCONF_VALUE_SCHEMA, tmp_list);                                      
 	g_slist_free(tmp_list);
}

void 
GConf::ChangeSet::set_pair(const String& key, int car_data, int cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_INT, GCONF_VALUE_INT,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, int car_data, bool cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_INT, GCONF_VALUE_BOOL,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, int car_data, double cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_INT, GCONF_VALUE_FLOAT,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, int car_data, const String& cdr_data)
{
	const char *tmp_cdr = cdr_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_INT, GCONF_VALUE_STRING,
	                          &car_data, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, int car_data, const Schema& cdr_data)
{
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                          &car_data, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, bool car_data, int cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_BOOL, GCONF_VALUE_INT,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, bool car_data, bool cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_BOOL, GCONF_VALUE_BOOL,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, bool car_data, double cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_BOOL, GCONF_VALUE_FLOAT,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, bool car_data, const String& cdr_data)
{
	const char *tmp_cdr = cdr_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_BOOL, GCONF_VALUE_STRING,
	                          &car_data, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, bool car_data, const Schema& cdr_data)
{
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_BOOL, GCONF_VALUE_SCHEMA,
	                          &car_data, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, double car_data, int cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_FLOAT, GCONF_VALUE_INT,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, double car_data, bool cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_FLOAT, GCONF_VALUE_BOOL,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, double car_data, double cdr_data)
{
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_FLOAT, GCONF_VALUE_FLOAT,
	                          &car_data, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, double car_data, const String& cdr_data)
{
	const char *tmp_cdr = cdr_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_FLOAT, GCONF_VALUE_STRING,
	                          &car_data, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, double car_data, const Schema& cdr_data)
{
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_INT, GCONF_VALUE_SCHEMA,
	                          &car_data, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, const String& car_data, int cdr_data)
{
	const char *tmp_car = car_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_STRING, GCONF_VALUE_INT,
	                          &tmp_car, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, const String& car_data, bool cdr_data)
{
	const char *tmp_car = car_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_STRING, GCONF_VALUE_BOOL,
	                          &tmp_car, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, const String& car_data, double cdr_data)
{
	const char *tmp_car = car_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_STRING, GCONF_VALUE_FLOAT,
	                          &tmp_car, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, const String& car_data, const String& cdr_data)
{
	const char *tmp_car = car_data.c_str();
	const char *tmp_cdr = cdr_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_STRING, GCONF_VALUE_STRING,
	                          &tmp_car, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, const String& car_data, const Schema& cdr_data)
{
	const char *tmp_car = car_data.c_str();
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_STRING, GCONF_VALUE_SCHEMA,
	                          &tmp_car, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, const Schema& car_data, int cdr_data)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_SCHEMA, GCONF_VALUE_INT,
	                          &tmp_car, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, const Schema& car_data, bool cdr_data)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_SCHEMA, GCONF_VALUE_BOOL,
	                          &tmp_car, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, const Schema& car_data, double cdr_data)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_SCHEMA, GCONF_VALUE_FLOAT,
	                          &tmp_car, &cdr_data);
}

void
GConf::ChangeSet::set_pair(const String& key, const Schema& car_data, const String& cdr_data)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	const char *tmp_cdr = cdr_data.c_str();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_SCHEMA, GCONF_VALUE_STRING,
	                          &tmp_car, &tmp_cdr);
}

void
GConf::ChangeSet::set_pair(const String& key, const Schema& car_data, const Schema& cdr_data)
{
	GConfSchema *tmp_car = car_data.gconf_schema();
	GConfSchema *tmp_cdr = cdr_data.gconf_schema();
	gconf_change_set_set_pair(gconf_change_set(), key.c_str(),
	                          GCONF_VALUE_SCHEMA, GCONF_VALUE_SCHEMA,
	                          &tmp_car, &tmp_cdr);
}

